/*
 * Copyright 2011 Matthias Butz <mtz@mtz.cc>. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without modification, are
 * permitted provided that the following conditions are met:
 *
 *   1. Redistributions of source code must retain the above copyright notice, this list of
 *      conditions and the following disclaimer.
 *
 *   2. Redistributions in binary form must reproduce the above copyright notice, this list
 *      of conditions and the following disclaimer in the documentation and/or other materials
 *      provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY MATTHIAS BUTZ ''AS IS'' AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL MATTHIAS BUTZ OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

package cc.mtz.sts.zza.sound;

import java.io.File;
import java.io.IOException;
import java.net.URL;
import java.net.URLEncoder;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Queue;
import java.util.concurrent.PriorityBlockingQueue;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.sound.sampled.AudioFormat;
import javax.sound.sampled.AudioInputStream;
import javax.sound.sampled.AudioSystem;
import javax.sound.sampled.LineUnavailableException;
import javax.sound.sampled.SourceDataLine;
import javax.sound.sampled.UnsupportedAudioFileException;

/**
 *
 * @author Matze
 */
public class SoundPlayer extends Thread {

    private final Queue<PlayEntry> toPlay = new PriorityBlockingQueue<PlayEntry>();
    private Map<String, String> replaces = new LinkedHashMap();
    private static SoundPlayer instance = new SoundPlayer();
    
    public static SoundPlayer getInstance() {
        return instance;
    }

    private SoundPlayer() {
        replaces.put("0([0-9]):", "$1:");
        replaces.put(":0([0-9])", ":$1");
        replaces.put("([0-9]*):([0-9]*) Uhr", "$1 Uhr $2");
        replaces.put("Uhr 0([^0-9]*)", "Uhr$1");
        replaces.put("ICE", "Ih zeh eh");
        replaces.put("IC", "Ih zeh");
        replaces.put("EC", "Eh zeh");
        replaces.put("THA", "Thalys");
        replaces.put("IRE", "Ih er eh");
        replaces.put("RE", "Er eh");
        replaces.put("RB", "Er beh");
        replaces.put("Hbf", "Hauptbahnhof");
        replaces.put("HB", "Hauptbahnhof");
        replaces.put("Bad Bf", "Badischer Bahnhof");
        replaces.put("Bf", "Bahnhof");
    }
    
    public void addText(Priority priority, String text) {
        for (Map.Entry<String, String> replace : replaces.entrySet()) {
            text = text.replaceAll(replace.getKey(), replace.getValue());
        }
        synchronized (toPlay) {
            toPlay.add(new PlayEntry(priority, text));
        }
    }
    
    public void addReplaces(Map<String, String> replaces) {
        this.replaces.putAll(replaces);
    }
    
    public boolean isIdle() {
        synchronized (toPlay) {
            return toPlay.isEmpty();
        }
    }
    
    @Override
    public void run() {
        for(;;) {
            AudioInputStream inStream = null;
            try {
                while (toPlay.isEmpty()) {
                    try {
                        synchronized (this) {
                            wait();
                        }
                    } catch (InterruptedException ex) {
                        Logger.getLogger(SoundPlayer.class.getName()).log(Level.SEVERE, null, ex);
                    }
                }
                String playText;
                synchronized (toPlay) {
                    playText = toPlay.poll().getText();
                }
                inStream = AudioSystem.getAudioInputStream(new URL("http://picard.mtztech.info/blechelse.php?text=" + URLEncoder.encode(playText, "ISO-8859-1")));
                AudioFormat	sourceFormat = inStream.getFormat();
                AudioFormat	targetFormat = new AudioFormat(
                        AudioFormat.Encoding.PCM_SIGNED,
                        sourceFormat.getSampleRate(),
                        16,
                        sourceFormat.getChannels(),
                        sourceFormat.getChannels() * 2,
                        sourceFormat.getSampleRate(),
                        false);
                AudioInputStream gongStream = AudioSystem.getAudioInputStream(new File("gong.wav"));
                AudioFormat gongFormat = gongStream.getFormat();
                AudioFormat gongTargetFormat = new AudioFormat(
                        AudioFormat.Encoding.PCM_SIGNED,
                        gongFormat.getSampleRate(),
                        16,
                        gongFormat.getChannels(),
                        gongFormat.getChannels() * 2,
                        gongFormat.getSampleRate(),
                        false);
                SourceDataLine gongLine = AudioSystem.getSourceDataLine(gongTargetFormat);
                gongLine.open();
                gongLine.start();
                byte[] buffer = new byte[128000];
                int bytesfilled = -1;
                do {
                    bytesfilled = gongStream.read(buffer);
                    if (bytesfilled > -1)
                        gongLine.write(buffer, 0, bytesfilled);
                } while (bytesfilled > -1);
                gongLine.drain();
                gongLine.stop();
                gongLine.close();
                gongStream.close();                
                SourceDataLine line = AudioSystem.getSourceDataLine(targetFormat);
                line.open();
                line.start();

                
                do {
                    bytesfilled = inStream.read(buffer);
                    if (bytesfilled > -1)
                        line.write(buffer, 0, bytesfilled);
                } while (bytesfilled > -1);
                line.drain();
                line.stop();
                line.close();
                inStream.close();
            } catch (LineUnavailableException ex) {
                Logger.getLogger(SoundPlayer.class.getName()).log(Level.SEVERE, null, ex);
            } catch (UnsupportedAudioFileException ex) {
                Logger.getLogger(SoundPlayer.class.getName()).log(Level.SEVERE, null, ex);
            } catch (IOException ex) {
                Logger.getLogger(SoundPlayer.class.getName()).log(Level.SEVERE, null, ex);
            } finally {
                try {
                    inStream.close();
                } catch (IOException ex) {
                    Logger.getLogger(SoundPlayer.class.getName()).log(Level.SEVERE, null, ex);
                }
            }
        }
    }
    
    public static enum Priority implements Comparable<Priority> {
        HIGH,
        MEDIUM,
        LOW;
    }
    
    private static class PlayEntry implements Comparable<PlayEntry> {
       
        private String text;
        private Priority priority;
        
        public PlayEntry(Priority priority, String text) {
            this.priority = priority;
            this.text = text;
        }
        
        public int compareTo(PlayEntry o) {
            return priority.compareTo(o.priority);
        }

        public String getText() {
            return text;
        }        
        
    }
    
}
