/*
 * Copyright 2011 Matthias Butz <mtz@mtz.cc>. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without modification, are
 * permitted provided that the following conditions are met:
 *
 *   1. Redistributions of source code must retain the above copyright notice, this list of
 *      conditions and the following disclaimer.
 *
 *   2. Redistributions in binary form must reproduce the above copyright notice, this list
 *      of conditions and the following disclaimer in the documentation and/or other materials
 *      provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY MATTHIAS BUTZ ''AS IS'' AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL MATTHIAS BUTZ OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package cc.mtz.sts.zza.listener;

import cc.mtz.sts.zza.AnsagenUtil;
import cc.mtz.sts.zza.AnschlussAnsage;
import cc.mtz.sts.zza.Main;
import cc.mtz.sts.zza.Rewriter;
import cc.mtz.sts.zza.sound.SoundPlayer;
import cc.mtz.sts.zza.StellwerkFile;
import cc.mtz.sts.zza.data.RewrittenDetails;
import cc.mtz.sts.zza.data.ZugBelegung;
import cc.mtz.sts.zza.data.DataCache;
import cc.mtz.sts.zza.data.Bahnhof;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.regex.Pattern;
import js.java.stspluginlib.PluginClient.ZugFahrplanZeile;

/**
 *
 * @author Matze
 */
public class AnsageManager implements StsListener {

    private StellwerkFile config;
    private Map<String, Map<String, Long>> verspaetungsAnsageZeiten = new HashMap();
    private Map<String, Set<String>> einfahrtGemeldet = new HashMap();
    private boolean einfahrten;
    private boolean anschluesse;
    private boolean verspaetungen;

    public AnsageManager(StellwerkFile config, boolean einfahrten,
            boolean anschluesse, boolean verspaetungen) {
        this.config = config;
        this.einfahrten = einfahrten;
        this.anschluesse = anschluesse;
        this.verspaetungen = verspaetungen;
    }

    public void bahnsteigSetup(String bahnsteig, Bahnhof bahnhof) {
        // ignore
    }

    public void naechsterZug(String gleis, Bahnhof bahnhof, int zid, DataCache cache) {
        // ignore
    }

    public void zugEinfahrt(String gleis, Bahnhof bahnhof, int zid, DataCache cache) {
        if (!einfahrtGemeldet.containsKey(gleis)
                || !einfahrtGemeldet.get(gleis).contains(cache.getDetailCache().get(zid).name)) {
            if (!einfahrtGemeldet.containsKey(gleis)) {
                einfahrtGemeldet.put(gleis, new HashSet());
            }
            einfahrtGemeldet.get(gleis).add(cache.getDetailCache().get(zid).name);
        }
        if (einfahrten) {
            RewrittenDetails details = Rewriter.getInstance().rewrite(cache.getDetailCache().get(zid));
            if (Pattern.matches(Main.IGNORE_PATTERN, details.name)) {
                return;
            }
            List<ZugFahrplanZeile> plan = cache.getFahrplaene().get(zid);
            if (Pattern.matches("^(TEC|CS|RoLa).*", cache.getDetailCache().get(zid).name)) {
                SoundPlayer.getInstance().addText(SoundPlayer.Priority.HIGH, "Auf Gleis "
                        + gleis.replaceFirst("[^0-9]*(.*)$", "$1")
                        + " Vorsicht, ein Zug f\u00e4hrt durch. "
                        + "Bitte zur\u00fcckbleiben.");
                synchronized (SoundPlayer.getInstance()) {
                    SoundPlayer.getInstance().notify();
                }
            } else if (Pattern.matches(bahnhof.getEndeRegex(), cache.getDetailCache().get(zid).nach) || 
                    Pattern.matches(bahnhof.getEndeRegex(), details.nach)) {               
                int startIndex = 0;
                for (int i = 0; i < plan.size(); i++) {
                    if (plan.get(i).gleis.equals(gleis)) {
                        startIndex = i + 1;
                        break;
                    }
                }
                SoundPlayer.getInstance().addText(SoundPlayer.Priority.HIGH, "Auf Gleis "
                        + gleis.replaceFirst("[^0-9]*(.*)$", "$1")
                        + " Einfahrt: " + AnsagenUtil.ansagenName(details.name)
                        + " von " + details.von + ". Ankunft "
                        + plan.get(startIndex - 1).getFormattedAn()
                        + " Uhr. Dieser Zug endet hier. Bitte nicht einsteigen. "
                        + "Vorsicht bei der Einfahrt!");
                synchronized (SoundPlayer.getInstance()) {
                    SoundPlayer.getInstance().notify();
                }
                if (anschluesse) {
                    long anschlussZeit = (plan.get(startIndex - 1).an - Main.client.getSimutime()) / 1000 + details.verspaetung * 60;
                    if (anschlussZeit < 0) {
                        anschlussZeit = 0;
                    }
                    Main.ses.schedule(new AnschlussAnsage(gleis, bahnhof, zid), anschlussZeit, TimeUnit.SECONDS);
                }
            } else {
                String vias = "";
                int startIndex = 0;
                for (int i = 0; i < plan.size(); i++) {
                    if (plan.get(i).gleis.equals(gleis)) {
                        startIndex = i + 1;
                        break;
                    }
                }
                if (details.rewritten) {
                    vias = details.vias;
                } else {
                    for (int i = startIndex; i < startIndex + 3; i++) {
                        String viaBhf = null;
                        if (viaBhf == null && plan.size() > i) {
                            vias += " - " + plan.get(i).gleis;
                        }
                        if (viaBhf != null && viaBhf.length() > 0) {
                            vias += " - " + viaBhf;
                        }
                    }
                    if (vias.length() > 0) {
                        vias = vias.substring(3);
                    }
                }
                String viaAnsage = "";
                if (vias.length() > 0) {
                    viaAnsage = " via " + vias;
                }
                StringBuilder ansagenText = new StringBuilder("Auf Gleis ");
                ansagenText.append(gleis.replaceFirst("[^0-9]*(.*)$", "$1"));
                if (Pattern.matches(bahnhof.getEndeRegex(), details.von)) {
                    ansagenText.append(" wird bereitgestellt: ");
                    ansagenText.append(AnsagenUtil.ansagenName(details.name));
                } else {
                    ansagenText.append(" Einfahrt: ");
                    ansagenText.append(AnsagenUtil.ansagenName(details.name));
                    ansagenText.append(" von ");
                    ansagenText.append(details.von);
                    ansagenText.append(" zur Weiterfahrt");
                    if (anschluesse) {
                        long anschlussZeit = (plan.get(startIndex - 1).an - Main.client.getSimutime()) / 1000 + details.verspaetung * 60;
                        if (anschlussZeit < 0) {
                            anschlussZeit = 0;
                        }
                        Main.ses.schedule(new AnschlussAnsage(gleis, bahnhof, zid), anschlussZeit, TimeUnit.SECONDS);
                    }
                }
                ansagenText.append(" nach ");
                ansagenText.append(details.nach);
                ansagenText.append(viaAnsage);
                ansagenText.append(". Abfahrt ");
                ansagenText.append(plan.get(startIndex - 1).getFormattedAb());
                ansagenText.append(" Uhr. Vorsicht bei der Einfahrt!");
                SoundPlayer.getInstance().addText(SoundPlayer.Priority.HIGH, ansagenText.toString());
                synchronized (SoundPlayer.getInstance()) {
                    SoundPlayer.getInstance().notify();
                }
            }
        }
    }

    public void idleAction(DataCache cache) {
        if (verspaetungen) {
            for (String gleis : cache.getBelegung().keySet()) {
                Bahnhof bahnhof = null;
                for (Bahnhof bhf : config.getBahnhoefe()) {
                    if (Pattern.matches(bhf.getGleiseRegex(), gleis)) {
                        bahnhof = bhf;
                        break;
                    }
                }
                if (bahnhof == null) {
                    throw new RuntimeException("Bahnhof not found");
                }
                long simutime = Main.client.getSimutime();
                List<ZugBelegung> planBelegung = cache.getBelegung().get(gleis);
                if (planBelegung != null) {
                    Collections.sort(planBelegung, new ZugBelegung.DelayComparator());
                    for (ZugBelegung zugBelegung : planBelegung) {
                        if (zugBelegung.zeile == null || zugBelegung.zug == null ||
                                Pattern.matches(Main.IGNORE_PATTERN, zugBelegung.zug.name)) {
                            continue;
                        }
                        if (((zugBelegung.zeile.an + zugBelegung.zug.verspaetung * 60000 > simutime
                                && ((zugBelegung.zeile.an - simutime) / 60000) <= 15)
                                || (zugBelegung.zeile.ab + zugBelegung.zug.verspaetung * 60000 > simutime
                                && ((zugBelegung.zeile.ab - simutime) / 60000) <= 15)) && zugBelegung.zug.verspaetung >= 5
                                && (!einfahrtGemeldet.containsKey(gleis) || !einfahrtGemeldet.get(gleis).contains(zugBelegung.zug.name))
                                && (!verspaetungsAnsageZeiten.containsKey(gleis)
                                || !verspaetungsAnsageZeiten.get(gleis).containsKey(zugBelegung.zug.name)
                                || (simutime - verspaetungsAnsageZeiten.get(gleis).get(zugBelegung.zug.name)) >= 300000)) {
                            RewrittenDetails details = Rewriter.getInstance().rewrite(zugBelegung.zug);
                            if (!verspaetungsAnsageZeiten.containsKey(gleis)) {
                                verspaetungsAnsageZeiten.put(gleis, new HashMap());
                            }
                            verspaetungsAnsageZeiten.get(gleis).put(details.name, simutime);
                            StringBuilder zugLauf = new StringBuilder();
                            String zeitText = "";
                            if (!(Pattern.matches(bahnhof.getEndeRegex(), details.von) || details.von.equals(""))) {
                                zugLauf.append(" von ");
                                zugLauf.append(details.von);
                                zeitText = "Ankunftszeit " + zugBelegung.zeile.getFormattedAn() + " Uhr";
                            }
                            if (!(Pattern.matches(bahnhof.getEndeRegex(), details.nach) || details.nach.equals(""))) {
                                zugLauf.append(" nach ");
                                zugLauf.append(details.nach);
                                zeitText = "Abfahrtzeit " + zugBelegung.zeile.getFormattedAb() + " Uhr";
                            }
                            SoundPlayer.getInstance().addText(SoundPlayer.Priority.LOW, "Auf Gleis " + gleis.replaceFirst("[^0-9]*(.*)$", "$1")
                                    + " Bitte beachten sie zu " + AnsagenUtil.ansagenName(details.name) + zugLauf.toString()
                                    + ", planm\u00e4\u00dfige " + zeitText + ": Dieser Zug wird heute voraussichtlich "
                                    + details.verspaetung + " Minuten sp\u00e4ter hier eintreffen. Wir bitten um Verst\u00e4ndnis.");
                            synchronized (SoundPlayer.getInstance()) {
                                SoundPlayer.getInstance().notify();
                            }
                        }
                    }
                }
            }
        }
    }
}
