/*
 * Copyright 2011 Matthias Butz <mtz@mtz.cc>. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without modification, are
 * permitted provided that the following conditions are met:
 *
 *   1. Redistributions of source code must retain the above copyright notice, this list of
 *      conditions and the following disclaimer.
 *
 *   2. Redistributions in binary form must reproduce the above copyright notice, this list
 *      of conditions and the following disclaimer in the documentation and/or other materials
 *      provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY MATTHIAS BUTZ ''AS IS'' AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL MATTHIAS BUTZ OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

package cc.mtz.sts.zza;

import cc.mtz.sts.zza.data.Bahnhof;
import cc.mtz.sts.zza.data.ZugRewrite;
import cc.mtz.sts.zza.data.RewrittenDetails;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLConnection;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.LinkedList;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.swing.text.DateFormatter;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.TransformerConfigurationException;
import javax.xml.transform.TransformerException;
import js.java.stspluginlib.PluginClient.ZugDetails;
import org.ccil.cowan.tagsoup.Parser;
import org.xml.sax.Attributes;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;
import org.xml.sax.helpers.DefaultHandler;

/**
 *
 * @author Matze
 */
public class Rewriter {
    
    private final boolean fallbackToRis;
    private static Rewriter instance;
    private StellwerkFile config;
    
    public static void init(StellwerkFile config, boolean fallbackToRis) {
        instance = new Rewriter(config, fallbackToRis);
    }
    
    public static Rewriter getInstance() {
        if (instance == null) {
            throw new IllegalStateException("Not initialized");
        } else {
            return instance;
        }
    }
    
    private Rewriter(StellwerkFile config, boolean fallbackToRis) {
        this.config = config;
        this.fallbackToRis = fallbackToRis;
    }
    
    public RewrittenDetails rewrite(ZugDetails details) {
        RewrittenDetails ret = new RewrittenDetails(details);
        boolean rewritten = false;
        for (ZugRewrite rewrite : config.getRewrites()) {
            if (rewrite.matches(details)) {
                rewritten = true;
                ret = rewrite.rewrite(details);
                if (rewrite.getVias() != null && rewrite.getVias().size() > 0) {
                    StringBuilder viaString = new StringBuilder();
                    for (String via : rewrite.getVias()) {
                        viaString.append(" - ");
                        viaString.append(via);
                    }
                    ret.vias = viaString.substring(3);
                }
                break;
            }
        }
        if (!rewritten && fallbackToRis) {
            try {
                RisParser parser = new RisParser();
                processRis(details.name, parser);
                if (parser.getStartBhf() != null && parser.getEndBhf() != null) {
                    ret.von = parser.getStartBhf();
                    ret.nach = parser.getEndBhf();
                    if (parser.getVias().size() > 1) {
                        // laufweg vor konfigurierten bahnhöfen entfernen
                        int startIndex = 0;
                        for (int i = 0; i < parser.getVias().size() - 1; i++) {
                            for (Bahnhof bahnhof : config.getBahnhoefe()) {
                                if (parser.getVias().get(i).equals(bahnhof.getName())) {
                                    startIndex = i + 1;
                                }
                            }
                        }
                        StringBuilder viaString = new StringBuilder();
                        int interval = (parser.getVias().size() - 1) / 4;
                        for (int i = startIndex; i < parser.getVias().size() - 1; i += interval) {
                            viaString.append(" - ");
                            viaString.append(parser.getVias().get(i));
                        }
                        if (viaString.length() > 3)
                            ret.vias = viaString.substring(3);
                    }
                    rewritten = true;
                } else {
                    Logger.getLogger(Rewriter.class.getName()).log(Level.WARNING, "Error fetching from RIS - Parsing not successful");
                }
            } catch (MalformedURLException ex) {
                Logger.getLogger(Rewriter.class.getName()).log(Level.WARNING, "Error fetching from RIS", ex);
            } catch (SAXException ex) {
                Logger.getLogger(Rewriter.class.getName()).log(Level.WARNING, "Error fetching from RIS", ex);
            } catch (IOException ex) {
                Logger.getLogger(Rewriter.class.getName()).log(Level.WARNING, "Error fetching from RIS", ex);
            } catch (TransformerConfigurationException ex) {
                Logger.getLogger(Rewriter.class.getName()).log(Level.WARNING, "Error fetching from RIS", ex);
            } catch (TransformerException ex) {
                Logger.getLogger(Rewriter.class.getName()).log(Level.WARNING, "Error fetching from RIS", ex);
            } catch (ParserConfigurationException ex) {
                Logger.getLogger(Rewriter.class.getName()).log(Level.WARNING, "Error fetching from RIS", ex);
            } catch (ParseException ex) {
                Logger.getLogger(Rewriter.class.getName()).log(Level.WARNING, "Error fetching from RIS", ex);
            }
            
        }
        ret.rewritten = rewritten;
        return ret;
    }
    
    private static void processRis(String zug, RisParser risHandler) throws ParseException, MalformedURLException, SAXException, IOException, TransformerConfigurationException, TransformerException, ParserConfigurationException {
        String productClass;
        if (zug.startsWith("ICE ")) {
            productClass = "1";
        } else if (zug.startsWith("IC ") || zug.startsWith("EC ")) {
            productClass = "2";
        } else if (zug.startsWith("IRE ")) {
            productClass = "8";
        } else if (zug.startsWith("IR ") || zug.startsWith("D ")) {
            productClass = "4";
        } else {
            productClass = "8";
        }
        DateFormatter df = new DateFormatter(new SimpleDateFormat("dd.MM.yy"));
        
        Calendar cal = Calendar.getInstance();
        cal.set(Calendar.DAY_OF_YEAR, cal.get(Calendar.DAY_OF_YEAR) + 1);
        URL url = new URL("http://mobile.bahn.de/bin/mobil/trainsearch.exe/dox?ld=96236&rt=1&use_realtime_filter=1&date=" + df.valueToString(cal.getTime()) + "&trainname=" + zug.replaceAll("[a-zA-Z ]", "") + "&stationFilter=80&start=Suchen&productClassFilter=" + productClass);
        URLConnection conn = url.openConnection();
        InputStream is = conn.getInputStream();
        ByteArrayOutputStream bos = new ByteArrayOutputStream();
        int val;
        do {
            val = is.read();
            if (val != -1) bos.write(val);
        } while (val != -1);
        try {
            is.close();
        } catch (Exception ex) {

        }

        Parser parser = new Parser();
        parser.setFeature(Parser.bogonsEmptyFeature, false);
        parser.setFeature(Parser.namespacesFeature, false);
        parser.setContentHandler(risHandler);
        parser.parse(new InputSource(new ByteArrayInputStream(bos.toByteArray())));
        parser = null;
        
        /*SAXSource ss = new SAXSource(parser, new InputSource(new ByteArrayInputStream(bos.toByteArray())));
        

        DocumentBuilderFactory dbf = DocumentBuilderFactory.newInstance();
        DocumentBuilder db = dbf.newDocumentBuilder();
        Document doc = db.newDocument();
        DOMResult dr = new DOMResult(doc);
        Transformer t = TransformerFactory.newInstance().newTransformer();
        t.transform(ss, dr);
        return doc;*/
    }
    
    private static class RisParser extends DefaultHandler {

        private static enum Mode {
            NONE,
            START_BHF,
            END_BHF
        }
        
        private StringBuilder chars = new StringBuilder();
        private Mode mode = Mode.NONE;
        private String startBhf;
        private String endBhf;
        private List<String> vias = new LinkedList();
        
        @Override
        public void characters(char[] ch, int start, int length) throws SAXException {
            chars.append(new String(ch, start, length));
        }

        @Override
        public void endElement(String uri, String localName, String qName) throws SAXException {
            if (qName.equals("td")) {
                //System.out.println("STOP");
                //System.out.println("chars: " + chars.toString());
                switch (mode) {
                    case START_BHF:
                        startBhf = chars.toString().trim();
                        break;
                    case END_BHF:
                        endBhf = chars.toString().trim();
                        vias.add(chars.toString().trim());
                        break;
                    default:
                }
            }
        }

        @Override
        public void startElement(String uri, String localName, String qName, Attributes attributes) throws SAXException {           
            if (qName.equals("td")) {
                chars = new StringBuilder();
                mode = Mode.NONE;
                if (attributes.getValue(uri, "class") != null && attributes.getValue(uri, "class").equals("station tqdetail top")) {
                    if (startBhf == null) {
                        //System.out.println("START_BHF");
                        mode = Mode.START_BHF;
                    } else {
                        //System.out.println("END_BHF");
                        mode = Mode.END_BHF;
                    }
                }
            }
        }

        public String getEndBhf() {
            return endBhf;
        }

        public String getStartBhf() {
            return startBhf;
        }

        public List<String> getVias() {
            return vias;
        }
        
    }

    
}
