/*
 * Copyright 2011 Matthias Butz <mtz@mtz.cc>. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without modification, are
 * permitted provided that the following conditions are met:
 *
 *   1. Redistributions of source code must retain the above copyright notice, this list of
 *      conditions and the following disclaimer.
 *
 *   2. Redistributions in binary form must reproduce the above copyright notice, this list
 *      of conditions and the following disclaimer in the documentation and/or other materials
 *      provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY MATTHIAS BUTZ ''AS IS'' AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL MATTHIAS BUTZ OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

package cc.mtz.sts.zza;

import cc.mtz.sts.zza.sound.SoundPlayer;
import cc.mtz.sts.zza.listener.StsListener;
import cc.mtz.sts.zza.data.ZugBelegung;
import cc.mtz.sts.zza.data.DataCache;
import cc.mtz.sts.zza.data.Bahnhof;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.regex.Pattern;
import js.java.stspluginlib.PluginClient;

/**
 *
 * @author Matze
 */
public class PluginClientImpl extends PluginClient {
    private final StellwerkFile config;
    private final List<StsListener> listener;
    private boolean ready = true;
    
    private DataCache currentCache;
    private DataCache lastRunCache;
    
    private Set<Integer> zuegeZuVerarbeiten;
    private Map<String, Set<String>> einfahrtGemeldet = new HashMap();
    

    public PluginClientImpl(String name, String author, String version, 
            String text, StellwerkFile config, List<StsListener> listener) {
        super(name, author, version, text);
        this.config = config;
        this.listener = listener;
        this.currentCache = new DataCache();
        SoundPlayer.getInstance().addReplaces(config.getSoundReplaces());
    }
    
    public DataCache getLastRunCache() {
        return new DataCache(lastRunCache);
    }

    @Override
    protected void connected() {
        request_bahnsteigliste();
    }

    @Override
    protected void closed() {
        System.exit(0);
    }

    @Override
    protected void response_anlageninfo(int aid, String name, String build) {
        /// wayne
    }

    @Override
    protected void response_bahnsteigliste(HashMap<String, HashSet<String>> bl) {
        List<String> keys = new ArrayList(bl.keySet());
        Collections.sort(keys);
        
        for (int bhf = 0; bhf < config.getBahnhoefe().size(); bhf++) {
            for (String bahnsteig : keys) {
                if (Pattern.matches(config.getBahnhoefe().get(bhf).getGleiseRegex(), bahnsteig)) {
                    for (StsListener callback : listener) {
                        callback.bahnsteigSetup(bahnsteig, config.getBahnhoefe().get(bhf));
                    }
                }
            }
        }
        
        if (ready) {
            ready = false;
            currentCache.clear();
            request_zugliste();
        }
    }

    @Override
    protected void response_zugliste(HashMap<Integer, String> zl) {
        zuegeZuVerarbeiten = new HashSet(zl.keySet());
        for (int zid : zl.keySet()) {
            request_zugdetails(zid);
        }
    }

    @Override
    protected void response_zugdetails(int zid, ZugDetails details) {
        currentCache.getDetailCache().put(zid, details);
        request_zugfahrplan(zid);
    }

    @Override
    protected void response_zugfahrplan(int zid, LinkedList<ZugFahrplanZeile> plan) {
        try {
            currentCache.getFahrplaene().put(zid, plan);
            zuegeZuVerarbeiten.remove(zid);
            for (ZugFahrplanZeile zeile : plan) {
                for (Bahnhof bahnhof : config.getBahnhoefe()) {
                    if (Pattern.matches(bahnhof.getGleiseRegex(), zeile.gleis)) {
                        List<ZugBelegung> gleisBelegung = currentCache.getBelegung().get(zeile.gleis);
                        if (gleisBelegung == null) {
                            gleisBelegung = new ArrayList();
                        }
                        gleisBelegung.add(new ZugBelegung(currentCache.getDetailCache().get(zid), zeile));
                        currentCache.getBelegung().put(zeile.gleis, gleisBelegung);
                    }
                }
            }
            if (zuegeZuVerarbeiten.isEmpty()) {
                //Logger.getLogger(getClass().getName()).log(Level.INFO, "Run finished");
                lastRunCache = new DataCache(currentCache);
                // check if audio scheduler is finished
                boolean soundIdle = SoundPlayer.getInstance().isIdle();
                long simutime = getSimutime();
                // schedule audioplay: einfahrten
                // if nothing todo at all: gebäck oder so
                for (String gleis : currentCache.getBelegung().keySet()) {
                    List<ZugBelegung> gleisBelegung = currentCache.getBelegung().get(gleis);
                    if (gleisBelegung != null) {
                        Collections.sort(gleisBelegung, new ZugBelegung.DelayComparator());
                        for (ZugBelegung zugBelegung : gleisBelegung) {
                            if (zugBelegung.zeile == null || zugBelegung.zug == null) {
                                continue;
                            }
                            if (zugBelegung.zeile.an + zugBelegung.zug.verspaetung * 60000 > simutime || zugBelegung.zeile.ab + zugBelegung.zug.verspaetung * 60000 > simutime) {
                                //this is our next train ;S
                                naechsterZug(zugBelegung.zug.zid, zugBelegung.zeile.gleis);
                                break;
                            }
                        }
                    }
                }
                // if audio scheduler was finished schedule audioplay: verspaetungen
                if (soundIdle) {
                    for (StsListener callback : listener) {
                        callback.idleAction(currentCache);
                    }
                }
                ready = true;
            }
        } catch (Exception ex) {
            Logger.getLogger(Main.class.getName()).log(Level.SEVERE, null, ex);
        }
    }

    public void naechsterZug(int zid, String gleis) {
        if (currentCache.getDetailCache().get(zid) == null) {
            return;
        }
        Bahnhof bahnhof = null;
        for (Bahnhof bhf : config.getBahnhoefe()) {
            if (Pattern.matches(bhf.getGleiseRegex(), gleis)) {
                bahnhof = bhf;
                break;
            }
        }
        if (bahnhof == null) {
            return;
        }
        for (StsListener callback : listener) {
            callback.naechsterZug(gleis, bahnhof, zid, currentCache);
        }
        if (currentCache.getDetailCache().get(zid).sichtbar && 
                currentCache.getDetailCache().get(zid).gleis.equals(gleis) && 
                (
                    !einfahrtGemeldet.containsKey(gleis) || 
                    !einfahrtGemeldet.get(gleis).contains(currentCache.getDetailCache().get(zid).name)
                )
           ) {
            for (StsListener callback : listener) {
                if (!einfahrtGemeldet.containsKey(gleis)) {
                    einfahrtGemeldet.put(gleis, new HashSet());
                }
                einfahrtGemeldet.get(gleis).add(currentCache.getDetailCache().get(zid).name);
                callback.zugEinfahrt(gleis, bahnhof, zid, currentCache);
            }
        }

    }

    public void requestZuege() {
        ready = false;
        currentCache.clear();
        request_zugliste();
    }

    public boolean isReady() {
        return ready;
    }

}
